#include <cmath>
#include <vector>

#include <SDL.h>
// #include <SDL_image.h>

#include <physfs.h>

#include <audiere.h>

#include "game.h"
#include "gamefont.h"

#include "mt19937ar.h"


#include "map.h"

#include "gfx.h"
#include "snd.h"

#include "object.h"
#include "object_manager.h"
#include "object_player.h"
#include "object_monster.h"
#include "object_item.h"


// Redraw screen:

void Game :: Draw(SDL_Surface *screen, const float alpha)
{
	viewport.screen	=	screen;
	switch( screen_state )
	{
	case STATE_SCREEN_GAME:		Draw_Game(screen,alpha);	break;
	case STATE_SCREEN_MENU:		Draw_Menu(screen,alpha);	break;
	case STATE_SCREEN_HELP:		Draw_Help(screen,alpha);	break;
	}
}

void Game :: Draw_Game(SDL_Surface *screen, const float alpha)
{
	map.Draw( &viewport, alpha );

	int xp = (this->mousestate->x / viewport.tilewidth) ;
	int yp = (this->mousestate->y / viewport.tileheight);

	if( (xp>=0) && (yp>=0) && (xp<map.width) && (yp<map.height) )
	{
		SDL_Rect rect;
		rect.x = xp * viewport.tilewidth;
		rect.y = yp * viewport.tileheight;
		SDL_BlitSurface( GFX.mouse[(anim>>1)&1], NULL, screen, &rect );
	}

	// Sidebar:
	int addy = -viewport.tileheight/2;
	float t=0.0f;

	if( mouse_tile_anim_countdown>0 )
	{
		t = (float)((mouse_tile_anim_countmax-mouse_tile_anim_countdown) + alpha) / (float)mouse_tile_anim_countmax;
		if(t>1.0f) t=1.0f;
		addy += -64 + (int)(64.0f * t );
	}

	for(int i=0; i<MAX_MOUSE_TILES; ++i)
	{
		SDL_Surface *bmp = map.GetTileBMP( mouse_tiles[MAX_MOUSE_TILES-1-i] );
		SDL_Rect rect;
		rect.x = (map.width+1) * viewport.tilewidth;
		rect.y = addy + (1 + 2*i) * viewport.tileheight;
		SDL_BlitSurface( bmp, NULL, screen, &rect );
	}
	{
		SDL_Surface *bmp = map.GetTileBMP( mouse_tile );
		SDL_Rect rect;

		moving_tile.x1		= (map.width+1) * viewport.tilewidth;
		moving_tile.y1		= addy + (1 + 2*MAX_MOUSE_TILES) * viewport.tileheight;

		rect.x = moving_tile.x1;
		rect.y = moving_tile.y1;
		SDL_BlitSurface( bmp, NULL, screen, &rect );
	
		rect.x = (map.width+1) * viewport.tilewidth - viewport.tilewidth/2;
		rect.y = (1 + 2*MAX_MOUSE_TILES) * viewport.tileheight - viewport.tileheight;
		SDL_BlitSurface( GFX.selection[0], NULL, screen, &rect );
	}

	if( (mouse_tile_anim_countdown>0) && moving_tile.active )
	{
		SDL_Rect rect;
		rect.x = (int)(t * (moving_tile.x2 - moving_tile.x1) + moving_tile.x1);
		rect.y = (int)(t * (moving_tile.y2 - moving_tile.y1) + moving_tile.y1);
		SDL_BlitSurface( moving_tile.bmp, NULL, screen, &rect );
	}

	{
		char buf[256];
		int xp =	(map.width+1) * viewport.tilewidth - viewport.tilewidth + viewport.tilewidth/4;
		int yp =	(1 + 2*MAX_MOUSE_TILES) * viewport.tileheight + 2*viewport.tileheight;

		sprintf(buf, "Level : %d\n\nTiles\nPlaced: %d\n\nCoins\nLeft  : %d", current_level+1, placed_tiles, coins_left );
		font->PrintText(screen, xp,yp,			buf );
	}

	if( message_delay > 0 )
	{
		int xp =	map.width * viewport.tilewidth/2 - 2*viewport.tilewidth;
		int yp =	map.height*viewport.tileheight / 2 - viewport.tileheight;
		
		font->PrintText(screen, xp,yp, msg );
	}

	font->PrintText(screen, 0, screen->h-16, game_copyright);
}

void Game::Update(const float dt)
{
	switch( screen_state )
	{
	case STATE_SCREEN_GAME:		Update_Game(dt);	break;
	case STATE_SCREEN_MENU:		Update_Menu(dt);	break;
	case STATE_SCREEN_HELP:		Update_Help(dt);	break;
	}
}

void Game::Update_Help(const float dt)
{
	if( this->keypressed[SDLK_ESCAPE] || this->keypressed[SDLK_SPACE] || 
		this->mousestate[1].mbpressed ||
		this->mousestate[2].mbpressed ||
		this->mousestate[3].mbpressed
		)
	{
		this->screen_state = STATE_SCREEN_MENU;
	}

	if( this->keypressed[SDLK_q] )
	{
		this->should_quit = true;
	}

	framecounter++;	
	framecounter &= 0x3FFFFFFF;

	// Clear keyboard/mouse
	ResetStates();
}

void Game::Draw_Help(SDL_Surface *screen, const float alpha)
{
	int offsetx = viewport.tilewidth;
	int yp = 24;

	font->PrintText(screen, offsetx, yp, "HELP MENU");

	const int tiletypes[] = {
		TileTypes::Special_AddPlayer,
		TileTypes::Special_Coin,
		0,
		TileTypes::Border,
		TileTypes::Wall,
		0,
		TileTypes::Ladder,
		TileTypes::OneWay_Left,
		TileTypes::OneWay_Right,
		TileTypes::Teleport,
		0,
		TileTypes::Special_Bomb,
		TileTypes::Skull,
		TileTypes::Special_AddEnemy,
		0,
	};

	const char * tilenames[] = {
		"The Player",
		"Coin",
		"-",
		"Indestructible brick",
		"Brick -\ndestructible by bomb",
		"-",
		"Ladder",
		"One-way left",
		"One-way right",
		"Teleports to other teleporters\nchoosen randomly",
		"-",
		"Bomb - explodes in 5 seconds",
		"Skull - instant death",
		"Monster - instant death",
		NULL
	};

	yp += 48;

	const int tile_height = viewport.tileheight + 8;

	for(int i=0; tilenames[i]; ++i)
	{
		if( tilenames[i][0] == '-' )
		{
			yp += 16;
		}
		else
		{
			SDL_Surface *bmp = map.GetTileBMP( tiletypes[i] );
			SDL_Rect rect;

			rect.x = offsetx;
			rect.y = yp;
			
			SDL_BlitSurface( bmp, NULL, screen, &rect );
		
			font->PrintText(screen, offsetx + viewport.tilewidth + 16, yp, tilenames[i] );
			yp += tile_height;
		}
	}

	const char * long_text = 
		"HOW TO PLAY\n\n\n"
		"You have to guide The Player(s)\n"
		"around the map by placing\n"
		"ladders and bricks for them to\n"
		"walk on.\n\n"
		"The Player(s) will walk by\n"
		"themselves in a straight line\n"
		"unless there are any obstacles\n"
		"or one-way signs in the way.\n\n"
		"But be careful: they will not\n"
		"avoid any obstacles such as\n"
		"skulls or monsters.\n\n"
		"Collect all coins to advance\n"
		"to the next level!"
		;

	font->PrintText(screen, offsetx + viewport.tilewidth*11, 24, long_text );

	font->PrintText(screen, 0, screen->h-16, game_copyright);
}


void Game::Update_Game(const float dt)
{
	bool no_update = false;

	if( message_delay> 0)
	{
		message_delay -= dt;
	}
	else
	{
		message_delay = 0;
		msg[0] = 0;
	}

	if( map.objmanager->GetMainObject() == NULL )
	{
		no_update = true;
		message_delay = 3000;
		sprintf(msg, "GAME OVER!");
		level_completed_delay = 0;
		level_completed = false;

		if( !game_over )
		{
			game_over = true;
			game_over_delay = 2000;
			SND.sound[5]->play();
		}
		else
		{
			game_over_delay -= dt;
			if( game_over_delay <0 )
			{
				game_over = false;
				game_over_delay = 0;
				//current_level = 0;
				InitNewGame();
				this->screen_state = STATE_SCREEN_MENU;
				menu_selected = 0;
			}
		}
	}
	else
	{
		if( level_completed )
		{
			no_update = true;
			if( level_completed_delay>0 )
			{
				level_completed_delay -= dt;
			}
			else
			{
				level_completed	=	false;
				++current_level;
				last_player_count = map.objmanager->GetObjectTypeCount( ObjectTypes::Player );

				InitNewGame();
			}
		}
		else
		{
			coins_left = map.objmanager->GetObjectSubTypeCount( ObjectTypes::Item, ItemTypes::Coins );
			if( coins_left == 0 )
			{
				message_delay = 3000;
				sprintf(msg, "LEVEL COMPLETED!");
				level_completed_delay = 3000;
				level_completed = true;
				no_update = true;
				SND.sound[7]->play();
			}
		}
	}

	if( !no_update )
	{
		int xp = (this->mousestate->x / viewport.tilewidth) ;
		int yp = (this->mousestate->y / viewport.tileheight);

		if( (xp>=0) && (yp>=0) && (xp<map.width) && (yp<map.height) )
		{
			if( this->mousestate[1].mbpressed )
			{
				int tile = map.GetTileTypeAt( xp, yp );
				if( (tile == TileTypes::None) || (tile == TileTypes::Open) )
				{
					{
						switch( mouse_tile )
						{
						case TileTypes::Special_AddPlayer:
							{
								Player *p = new Player();
								
								p->x = xp;
								p->y = yp;
								p->oldx = p->x;
								p->oldy = p->y;
								p->update_counter = genrand_int31() % p->update_counter_max;

								map.objmanager->LinkObject( p );
							}
							break;
						case TileTypes::Special_AddEnemy:
							{
								Monster *p = new Monster();
								
								p->x = xp;
								p->y = yp;
								p->oldx = p->x;
								p->oldy = p->y;
								p->update_counter = genrand_int31() % p->update_counter_max;

								map.objmanager->LinkObject( p );
							}
							break;
						case TileTypes::Special_Bomb:
							{
								ObjectItem *p = new ObjectItem();
								
								p->x = xp;
								p->y = yp;
								p->oldx = p->x;
								p->oldy = p->y;
								p->update_counter = genrand_int31() % p->update_counter_max;
								p->timeout_counter = 5;
								p->timeout_delay = 1000;
								p->itemtype = ItemTypes::Bomb;

								map.objmanager->LinkObject( p );
							}
							break;
						default:
							map.PasteTileAt( xp, yp, mouse_tile );
							break;
						}

						GenerateNewMouseTiles();
						++placed_tiles;
						//SND.sound[8]->play();
					}
				}
			}
		}
	}

	anim_delay += dt;
	if( anim_delay >= anim_delay_max )
	{
		anim_delay -= anim_delay_max;
		anim = (anim + 1) & 7;
	}

	if( mouse_tile_anim_countdown>0 ) 
	{
		mouse_tile_anim_countdown--;
	}
	else
	{
		moving_tile.active = false;
	}

	if( !no_update )
	{
		map.Update( dt );
	}
	else
	{
		map.UpdateAnims( dt );
	}

	if( this->keypressed[SDLK_ESCAPE] || this->keypressed[SDLK_q] )
	{
		this->screen_state = STATE_SCREEN_MENU;
	}

	if( this->keypressed[SDLK_h] || this->keypressed[SDLK_F1] )
	{
		screen_state = STATE_SCREEN_HELP;
	}

	// Cheat mode:
	/*
	if( this->keypressed[SDLK_1] )	mouse_tile = TileTypes::Wall;
	if( this->keypressed[SDLK_2] )	mouse_tile = TileTypes::Ladder;
	if( this->keypressed[SDLK_3] )	mouse_tile = TileTypes::OneWay_Left;
	if( this->keypressed[SDLK_4] )	mouse_tile = TileTypes::OneWay_Right;
	if( this->keypressed[SDLK_5] )	mouse_tile = TileTypes::Skull;
	if( this->keypressed[SDLK_6] )	mouse_tile = TileTypes::Teleport;
	if( this->keypressed[SDLK_7] )	mouse_tile = TileTypes::Special_AddEnemy;
	if( this->keypressed[SDLK_8] )	mouse_tile = TileTypes::Special_Bomb;
	if( this->keypressed[SDLK_9] )	mouse_tile = TileTypes::Border;
	*/

	// Clear keyboard/mouse
	ResetStates();

	framecounter++;
	framecounter &= 0x3FFFFFFF;
}

// Menu Stuff:

void Game::Draw_Menu(SDL_Surface *screen, const float alpha)
{
	int offsetx = screen->w / 2 - viewport.tilewidth*4;
	int yp = 8 + screen->h / 4;

	font->PrintText(screen, offsetx, yp, "MAIN MENU");

	// draw menus:
	const char *menu_str[] = {"Continue", "New Game", "Help", "Exit", NULL};
	const int menu_height = 16 + 16;

	yp += 64;

	for(int i=0; menu_str[i]; ++i)
	{
		font->PrintText(screen, offsetx, yp + i * menu_height, menu_str[i] );
	}

	// nice menu selection:
	if( menu_selected>= 0)
	{
		SDL_Rect rect;
		int xc = offsetx + 9*14/2 - 8;
		int yc = yp + menu_selected * menu_height + 6;
		
		const int max_stars = 64;
			
		float t2 = ((float)(framecounter&63) + alpha) / 64.0f;
		float t3 = 1.0f - ((float)(framecounter&127) + alpha) / 128.0f;

		for(int i=0; i<max_stars; ++i)
		{
			float t = (float)i / (float)max_stars;
			float angle = (3.141592653589793f*2.0f) * (t + t2);	

			float amp = 1.0f + 0.0625f * sinf( (3.141592653589793f*2.0f) * (t*1.0f+t3*1.0f) * 8.0f );

			rect.x = xc + (int)( amp * (2*9*14/3) * cosf(angle) );
			rect.y = yc + (int)( amp * (16.0f) * sinf(angle) );

			SDL_BlitSurface( GFX.stars[i&3], NULL, screen, &rect );
		}
	}

	font->PrintText(screen, 0, screen->h-16, game_copyright);
}

void Game::Update_Menu(const float dt)
{
	if( this->keypressed[SDLK_UP] )
	{
		this->menu_selected--;
	}

	if( this->keypressed[SDLK_DOWN] )
	{
		this->menu_selected++;
	}

	// check mouse:
	
	//menu_selected = -1;
	if( viewport.screen )
	{
		int menu_left   = viewport.screen->w / 2 - viewport.tilewidth*4;
		int menu_top    = 8 + viewport.screen->h / 4 + 64;

		int menu_width  = 9*14;
		int menu_height = 16 + 16;
		
		int mx = mousestate->x;
		int my = mousestate->y;

		if( (mx >= menu_left) && (mx < menu_left + menu_width) )
		{
			if( (my >= menu_top) && (my < menu_top + 4*menu_height) )
			{
				menu_selected = (my - menu_top) / menu_height;
			}
		}
	}

	if( menu_selected<0 ) menu_selected = 0;
	else if( menu_selected > 3) menu_selected = 3;

	/*
		int offsetx = ;
		int yp = 8 + screen->h / 4;
		
		int xc = offsetx + 9*14/2 - 8;
		int yc = yp + menu_selected * menu_height + 6;
		
		(2*9*14/3)
		h=(16.0f)
	*/


	if( this->keypressed[SDLK_RETURN] || mousestate[1].mbpressed )
	{
		switch( menu_selected )
		{
		case 0:	screen_state = STATE_SCREEN_GAME;	break;
		case 1: screen_state = STATE_SCREEN_GAME;	
			current_level = 0;
			this->last_player_count = 1;
			InitNewGame();
			break;
		case 2:	screen_state = STATE_SCREEN_HELP;	break;
		case 3:	this->should_quit = true;
			break;
		default: break;
		}
	}

	if( this->keypressed[SDLK_ESCAPE] || this->keypressed[SDLK_q] )
	{
		this->should_quit = true;
	}

	if( this->keypressed[SDLK_h] || this->keypressed[SDLK_F1] )
	{
		screen_state = STATE_SCREEN_HELP;
	}

	// Clears keyboard/mouse:
	ResetStates();

	framecounter++;
	framecounter &= 0x3FFFFFFF;
}


// Input handling:  SDLinput -> game wrapper

void Game::InitStates()
{
	for(int i=0; i<MAX_KEYSTATES; i++)
	{
		keypressed[i]	= false;
		keydown[i]		= false;
	}

	for(int i=0; i<MAX_MOUSESTATES; i++)
	{
		mousestate[i].mbpressed	= false;
		mousestate[i].button	= 0;
		mousestate[i].mbdown	= false;
		mousestate[i].moved		= false;
		mousestate[i].x			= 0;
		mousestate[i].y			= 0;
	}
}

void Game::ResetStates()
{
	for(int i=0; i<MAX_KEYSTATES; i++)
	{
		keypressed[i] = false;
	}

	for(int i=0; i<MAX_MOUSESTATES; i++)
	{
		mousestate[i].mbpressed	= false;
	}
}

void Game::KeyDown		(SDL_keysym *key)
{
	int i = key->sym;
	if( i < MAX_KEYSTATES )
	{
		keypressed[i]	= true;
		keydown[i]		= true;
	}
}

void Game::KeyUp		(SDL_keysym *key)
{
	int i = key->sym;
	if( i < MAX_KEYSTATES )
	{
		keydown[i]		= false;
	}
}

void Game::MouseMove	(SDL_MouseMotionEvent *motion)
{
	mousestate[0].moved = true;
	mousestate[0].x	= motion->x;
	mousestate[0].y = motion->y;
}

void Game::MouseDown	(SDL_MouseButtonEvent *button)
{
	int i = button->button;
	if( (i>=1) && (i<MAX_MOUSESTATES) )
	{
		mousestate[i].mbpressed = true;
		mousestate[i].mbdown	= true;
		mousestate[i].x			= button->x;
		mousestate[i].y			= button->y;
	}
}

void Game::MouseUp		(SDL_MouseButtonEvent *button)
{
	int i = button->button;
	if( (i>=1) && (i<MAX_MOUSESTATES) )
	{
		mousestate[i].mbdown	= false;
		mousestate[i].x			= button->x;
		mousestate[i].y			= button->y;
	}
}

// constructor/destructor
Game::Game()
	: framecounter(0), anim(0), anim_delay(0), anim_delay_max(0), mouse_tile(0),
	mouse_tile_anim_countdown(0), mouse_tile_anim_countmax(4),
	current_level(0), placed_tiles(0), coins_left(0), menu_selected(1),
	last_player_count(1),

	message_delay(0),
	level_completed(false),
	level_completed_delay(0),
	game_over(false),
	game_over_delay(0)
{
	// Virtual filesystem:
	PHYSFS_init(NULL);
	PHYSFS_addToSearchPath(".", 0);
	PHYSFS_addToSearchPath("gamedata.dat", 1);

	// graphics:
	// global class:
	GFX.LoadBitmaps();
	InitRandomTiles();
	msg[0] = 0;

	SND.OpenDevice();
	SND.LoadSounds();

	font = new GameFont(); 
	font->SetSDLSurface( LoadImage("gamedata/font.bmp",	true, 0x000000) );
	font->SetCharWidthHeight( 16, 16 );
	font->SetKerning(12);

	// global font:
	GFX.font = font;

	viewport.tileheight	=	32;
	viewport.tilewidth	=	32;

	memset( &moving_tile, 0, sizeof(moving_tile) );

	map.AllocateMap(20,18);

	InitStates();
	InitNewGame();
	screen_state = STATE_SCREEN_MENU;

	// background music:
	SND.music[0]->setRepeat(true);
	SND.music[0]->play();
}

Game::~Game()
{
	PHYSFS_deinit();

	delete font;
}

// Game logic:
void Game::InitNewGame()
{
	this->game_isdone = false;
	this->should_quit = false;

	this->framecounter = 0;
	this->anim = 0;
	this->anim_delay = 0;
	this->anim_delay_max = 0.3;

	game_over = false;
	game_over_delay = 0;
	level_completed = false;
	level_completed_delay = 0;

	this->mouse_tile = TileTypes::None;

	if( last_player_count < 1 ) last_player_count = 1;

	map.InitNewGame(current_level, last_player_count);
	GenerateAllMouseTiles();

	memset( &moving_tile, 0, sizeof(moving_tile) );

	//current_level = 0;
	placed_tiles	=	0;

	this->coins_left = map.objmanager->GetObjectSubTypeCount( ObjectTypes::Item, ItemTypes::Coins );

	message_delay = 5000;
	sprintf(msg, "LEVEL %d:\n\nCOLLECT %d COINS!", current_level+1, coins_left );

	ResetStates();
}

void Game :: GenerateAllMouseTiles()
{
	mouse_tile = 0;
	memset( mouse_tiles, 0, sizeof(mouse_tiles) );
	for(int i=0; i<MAX_MOUSE_TILES+1; ++i)
	{
		GenerateNewMouseTiles();
	}
	mouse_tile_anim_countdown = 0;
}

void Game :: GenerateNewMouseTiles()
{
	mouse_tile = mouse_tiles[0];
	for(int i=0; i<MAX_MOUSE_TILES-1; ++i)
	{
		mouse_tiles[i] = mouse_tiles[i+1];
	}
	
	mouse_tiles[ MAX_MOUSE_TILES-1 ] = random_tiles[genrand_int31() % random_tiles.size()];

	mouse_tile_anim_countdown += mouse_tile_anim_countmax;
}

void Game :: InitRandomTiles()
{
	const int tile_lookup[] = {
		TileTypes::Wall,				10,
		TileTypes::Ladder,				10,
		TileTypes::Border,				3,
		TileTypes::OneWay_Left,			2,
		TileTypes::OneWay_Right,		2,
		TileTypes::Skull,				4,
		TileTypes::Teleport,			2,
		TileTypes::Special_AddEnemy,	2,
		TileTypes::Special_Bomb,		2,
		TileTypes::Special_AddPlayer,	1,
		0,0
	};

	this->random_tiles.clear();

	for(int i=0; tile_lookup[i]; i += 2 )
	{
		for(int j=0; j<tile_lookup[i+1]; ++j)
		{
			random_tiles.push_back( tile_lookup[i] );
		}	
	}


}


