#include <ctime>

#include <SDL.h>
#include <physfs.h>
#include <audiere.h>

#include "mt19937ar.h"

#include "game.h"


void GameLoop(SDL_Surface * screen, const Uint32 dt, int delay)
{
	Uint32	time_accumulator = 0, 
			time_current=0,
			time_new=0, 
			time_delta=0;
	int screenshot_counter = 0;

	const float dt_f = (float)dt;
	const float dt_inv = 1.0f / dt_f;

	SDL_Rect rect;
	SDL_Event event;
	bool should_quit = false;
	
	Game game;

	event.type = 0;

	while( !should_quit )
	{
		while ( SDL_PollEvent(&event) )
		{
			if ( (event.type == SDL_QUIT)  || (game.ShouldQuit()) )
			{
				should_quit = true;
			}
			
			if( event.type == SDL_KEYDOWN )
			{
				if( event.key.keysym.sym == SDLK_F11 )
				{
					char buf[64]={0};
					sprintf(buf, "screenshot%.4d.bmp", screenshot_counter++ );
					SDL_SaveBMP( screen, buf );
				}
			}

			switch( event.type )
			{
			case SDL_KEYDOWN:			game.KeyDown	( &event.key.keysym );	break;
			case SDL_KEYUP:				game.KeyUp		( &event.key.keysym );	break;

			case SDL_MOUSEMOTION:		game.MouseMove	( &event.motion );		break;
			case SDL_MOUSEBUTTONDOWN:	game.MouseDown	( &event.button );		break;
			case SDL_MOUSEBUTTONUP:		game.MouseUp	( &event.button );		break;
			}
		}

		//	Run game logic at constanst fps:
		time_current	= time_new;
		time_new		= SDL_GetTicks();
		time_delta		= (time_new - time_current);

		if( time_delta > 1000 ) time_delta = 1000;

		time_accumulator += time_delta;
		while( time_accumulator >= dt )
		{
			game.Update( dt_f );
			time_accumulator -= dt;
		}

		const float framealpha = (float)time_accumulator * dt_inv;

		rect.w = (Uint16)screen->w;
		rect.h = (Uint16)screen->h;
		rect.x = 0;
		rect.y = 0;
		SDL_FillRect(screen, &rect, 0);

		game.Draw( screen, framealpha );
		
		SDL_Flip(screen);

		// dirty fix to prevent 100% CPU hog:
		if( delay>0 )	SDL_Delay( delay );
 	}
}



int main(int argc, char *argv[])
{
	// initialize Mersenne Twister random number generator:
	init_genrand( (unsigned long)time(NULL) );

    // Initialize the SDL library
    if ( SDL_Init(SDL_INIT_VIDEO) < 0 )
	{
		printf("SDL_Init() failed!\n");
        // Clean up on exit
        SDL_Quit();
	}  

	// commandline args
	int width		= 800;
	int height		= 600;
	int bits		= 0;
	int delay		= 1;
	Uint32 dt		= 80;	// game-speed
	Uint32 flags	= SDL_DOUBLEBUF | SDL_HWSURFACE;	

	for(int i=1; i<argc; ++i)
	{
		if( argv[i][0] == '-' )
		{
			switch( argv[i][1] )
			{
			case 'w':	i++; width	= atoi(argv[i]);	break;	
			case 'h':	i++; height = atoi(argv[i]);	break;	
			case 'b':	i++; bits	= atoi(argv[i]);	break;	
			case 's':	i++; delay	= atoi(argv[i]);	break;
			
			case 'f':	flags	|= SDL_FULLSCREEN;	break;	
			case 'd':	i++; dt		= atoi(argv[i]);	break;
			default: break;
			}
		}
	}

	/* Get available fullscreen/hardware modes */
	SDL_Rect ** modes = SDL_ListModes(NULL, flags);

	/* Check is there are any modes available */
	if(modes == (SDL_Rect **)0)
	{
		printf("No modes available!\n");
		exit(-1);
	}

	/* Check if our resolution is restricted */
	if(modes == (SDL_Rect **)-1)
	{
		printf("All resolutions available.\n");
	}
	else
	{
		/* Print valid modes */
		int i;
		printf("Available Modes:\n");
		for(i=0;modes[i];++i)
		{
			printf("  %d x %d\n", modes[i]->w, modes[i]->h);
		}
	}

    SDL_Surface * screen = SDL_SetVideoMode(width,height,bits,flags);
    
	if ( screen == NULL )
	{
		printf("SDL_SetVideoMode() failed!\n");
        SDL_Quit();
	}

	SDL_WM_SetCaption( game_copyright, NULL );

	if( dt < 1 ) dt = 1;
	GameLoop(screen, (const Uint32)dt, delay);

	SDL_Quit();
	return 0;
}

