#include <cstdlib>
#include <cassert>
#include <vector>

#include <SDL.h>

#include "object_manager.h"
#include "map.h"
#include "viewport.h"

#include "object.h"
#include "object_player.h"
#include "object_item.h"

#include "gfx.h"

#include "mt19937ar.h"

void Map :: AllocateMap(int w, int h)
{
	if( tiles )
	{
		free(tiles); tiles=NULL;
	}

	width	= w;
	height	= h;

	tiles = (maptile_t *)calloc( width * height, sizeof(maptile_t) );
}

void Map :: PasteTileAt(int x, int y, int tile)
{
	if( (tiles == NULL) ||
		(x<0) || (y<0) || (x>=width) || (y >= height ) )	
		return ;

	 tiles[ (y*width)+x ].tile = tile;	
}

int Map :: GetTileTypeAt(int x, int y)
{
	if( (tiles == NULL) || (width<1) || (height<1) )		return TileTypes::Invalid;

	if( (x<0) || (y<0) || (x>=width) || (y >= height ) )	return TileTypes::Outside;

	return tiles[ (y*width)+x ].tile;	
}


void Map :: Update(const float dt)
{
	objmanager->Update( dt );
}

void Map :: UpdateAnims(const float dt)
{
	objmanager->UpdateAnims( dt );
}


SDL_Surface * Map :: GetTileBMP( int tiletype )
{
	switch( tiletype )
	{
	case TileTypes::Border:				return GFX.stones[0];	break;
	case TileTypes::Wall:				return GFX.stones[1];	break;
	case TileTypes::Ladder:				return GFX.ladder[0];	break;
	case TileTypes::Skull:				return GFX.skull[0];	break;
	case TileTypes::OneWay_Left:		return GFX.tiles[0];	break;
	case TileTypes::OneWay_Right:		return GFX.tiles[1];	break;
	case TileTypes::Teleport:			return GFX.tiles[2];	break;
	case TileTypes::Special_AddPlayer:	return GFX.player[0];	break;
	case TileTypes::Special_AddEnemy:	return GFX.monster[1];	break;
	case TileTypes::Special_Bomb:		return GFX.tiles[3];	break;
	case TileTypes::Special_Coin:		return GFX.coin[0];		break;
	default:	return NULL;	break;
	}
	//return NULL;
}

void Map :: Draw(Viewport *viewport, const float alpha)
{
	SDL_Rect rect;
	SDL_Surface *bmp;

	maptile_t * tileptr = tiles;

	for(int y=0; y<height; ++y)
	{
		for(int x=0; x<width; ++x)
		{	
			rect.x = viewport->offsetx + (viewport->tilewidth) * x;
			rect.y = viewport->offsety + (viewport->tileheight) * y;

			bmp = GetTileBMP( tileptr->tile );

			if(bmp) SDL_BlitSurface( bmp, NULL, viewport->screen, &rect );

			++tileptr;
		}

	}

	objmanager->Draw( viewport, alpha );
}

void Map :: ExplosionAt(int x, int y,int radius)
{
	if( (tiles==NULL) || (width<1) || (height<1) ) return;

	for(int yp=-radius; yp<=radius; ++yp)
	{
		for(int xp=-radius; xp<=radius; ++xp)
		{	
			int ny = y + yp;
			int nx = x + xp;
			if( (nx>=0) && (ny>=0) && (nx<width) && (ny<height) )
			{
				int pos = ny * width + nx;
				if( tiles[pos].tile != TileTypes::Border )
				{
					tiles[pos].tile = TileTypes::None;
				}
			}
		}
	}

	this->objmanager->ExplosionAt(x,y,radius);
}


Map :: Map()
	:	width(0), height(0), tiles(NULL)
{
	objmanager = new ObjectManager(this);
}

Map :: ~Map()
{
	if( tiles ) free(tiles);
	delete objmanager;
}

void Map :: InitNewGame(int level, int player_count)
{
	assert( tiles );
	assert( width>0 );
	assert( height>0 );

	objmanager->Clear();

	memset( tiles, 0, width * height * sizeof(maptile_t) );

	// border:
	for(int x=0; x<width; ++x)
	{
		tiles[x].tile = TileTypes::Border;
		tiles[(height-1)*width + x].tile = TileTypes::Border;
	}
	// border:
	for(int y=0; y<height; ++y)
	{
		tiles[y*width].tile = TileTypes::Border;
		tiles[(y)*width + width - 1].tile = TileTypes::Border;
	}


	int count = level + genrand_int31() % (5 + level);

	// Random indestructible stones 
	for(int i=0; i<count; ++i)
	{
		int x = 1 + genrand_int31() % (width-2);
		int y = 1 + genrand_int31() % (height-3);
		tiles[(y)*width + x].tile = TileTypes::Border;
	}

	// teleporters:
	count = -2 + level/3 + genrand_int31() % (1 + level);
	for(int i=0; i<count; ++i)
	{
		int x = 1 + genrand_int31() % (width-2);
		int y = 1 + genrand_int31() % (height-5);
		tiles[(y)*width + x].tile = TileTypes::Skull;
	}

	// skulls, instant death:
	count = level/3 + genrand_int31() % (2 + level);
	for(int i=0; i<count; ++i)
	{
		int x = 1 + genrand_int31() % (width-2);
		int y = 1 + genrand_int31() % (height-5);
		tiles[(y)*width + x].tile = TileTypes::Skull;
	}

	// some coins:

	count = 5 + level + genrand_int31() % (1+level/2);

	for(int i=0; i<count; ++i)
	{
		ObjectItem *p = new ObjectItem();
		p->itemtype = ItemTypes::Coins;
		p->coins = 1;

		p->x = 1 + genrand_int31() % (width-2);
		p->y = 1 + genrand_int31() % (height-3);
		p->oldx = p->x;
		p->oldy = p->y;

		objmanager->LinkObject( p );

		tiles[(p->y)*width + p->x].tile = TileTypes::None;

	}

	// Player:
	if( player_count < 1 ) player_count = 1;
	{
		for(int i=0; i<player_count; ++i)
		{
			Player *p = new Player();

			p->x = ((1 + (width-2) ) * (i+1)) / (player_count+1);
			p->y = height - 2;
			p->oldx = p->x;
			p->oldy = p->y;

			objmanager->LinkObject( p );

			tiles[(p->y)*width + p->x].tile = TileTypes::None;

			objmanager->main_object = p;
			p->update_counter = i * p->update_counter_max / player_count;
		}
	}
}

bool Map :: GetTeleportDestination( int *xp, int *yp )
{
	assert( tiles );
	assert( width>0 );
	assert( height>0 );
	//if( (tiles==NULL) || (width<1) || (height<1) ) return false;

	std::vector<int>	locations;

	locations.clear();
	int count = 0;

	for(int y=0; y<height; ++y)
	{
		for(int x=0; x<width; ++x)
		{
			int tiletype = this->GetTileTypeAt(x,y);
			if( (tiletype == TileTypes::Teleport) && !((x == (*xp)) && (y == (*yp))) )
			{
				int pos = (y * width + x);
				locations.push_back( pos );
				count++;
			}
		}
	}

	if( count>0 )
	{
		int index = genrand_int31() % count;

		int destpos = locations[index];

		*xp = destpos % width;
		*yp = destpos / width;

		return true;
	}

	return false;
}

