#include <cassert>
#include <cmath>

#include <SDL.h>

#include "map.h"

#include "object_manager.h"
#include "object.h"
#include "object_player.h"
#include "object_item.h"

#include "viewport.h"
#include "gfx.h"
#include "snd.h"

class ItemStates
{
public:
	enum
	{
		None = 0,

		Idle,
		Exploding,

		Last
	};
};

void ObjectItem :: CollisionWith( Object * other )
{
	switch( this->itemtype )
	{
	case ItemTypes::Coins:
		if( other->GetType() == ObjectTypes::Player )
		{
			Player *p = (Player *)other;

			p->life		+=	life;
			p->keys		+=	keys;
			p->coins	+=	coins;
			p->score	+=	score;

			life = keys = coins = score = 0;
			this->delete_me = true;
			SND.sound[3]->play();
		}
		break;
	case ItemTypes::Bomb:
		{
			// explode or not explode?

		}
	default: break;
	}
}

// death by explosion:
void ObjectItem :: Death()
{
	switch( this->itemtype )
	{
	case ItemTypes::Bomb:
		{
			// explode!
			timeout_counter = 0;

		}
	default: break;
	}	
}

void ObjectItem :: Update(const float dt)
{
	anim_delay += dt;
	if( anim_delay >= anim_delay_max )
	{
		anim_delay -= anim_delay_max;
		anim = (anim + 1) & 7;
	}

	++update_counter;
	if( update_counter >= update_counter_max )
	{
		if( state == ItemStates::Exploding )
		{
			this->delete_me = true;
		}

		oldx = x;
		oldy = y;

		update_counter -= update_counter_max;

		switch( this->itemtype )
		{
		case ItemTypes::Bomb:
			{
				--timeout_counter;
				if( (timeout_counter < 0) && (state != ItemStates::Exploding) )
				{
					// Explode!
					state = ItemStates::Exploding;
					assert( parent );
					assert( parent->GetMap() );
					this->parent->GetMap()->ExplosionAt( this->x, this->y, 1 );

					SND.sound[1]->play();
					//this->delete_me = true;
				}
			}
			break;

		default: break;
		}
	}
}

void ObjectItem :: UpdateAnims(const float dt)
{
	anim_delay += dt;
	if( anim_delay >= anim_delay_max )
	{
		anim_delay -= anim_delay_max;
		anim = (anim + 1) & 7;
	}

	++update_counter;
	if( update_counter >= update_counter_max )
	{
		oldx = x;
		oldy = y;

		update_counter -= update_counter_max;

	}
}

void ObjectItem :: Draw(Viewport *viewport, const float alpha)
{
	const int direction_lookup[] = {0,0,1};
	SDL_Rect rect;
	SDL_Surface *bmp = NULL;

	float t = ((float)update_counter + (float)alpha) / (float)update_counter_max;
	if( t>1.0f) t = 1.0f;

	float xp = t * (float)(x - oldx) + oldx;
	float yp = t * (float)(y - oldy) + oldy;

	rect.x = viewport->offsetx + (int)((float)viewport->tilewidth  * xp);
	rect.y = viewport->offsety + (int)((float)viewport->tileheight * yp);
		
	switch( itemtype )
	{
	case ItemTypes::Coins:	
		bmp = GFX.coin[0];	
		rect.x += viewport->tilewidth/4;
		rect.y += viewport->tileheight/4;
		break;
	case ItemTypes::Bomb:	
		{
			if( state == ItemStates::Exploding )
			{
				SDL_Rect r;
				const int max_stars = 48;
				const int maxlevels = 3;
				for(int levels=0; levels<maxlevels; ++levels)
				{
					float amp = 64.0f * (0.25f + (float)levels / (float)maxlevels);
					for(int i=0; i<max_stars; ++i)
					{
						float v = (float)i/(float)max_stars;
						float a = amp + 8.0f * sinf( 6.0f * (3.141592653589793f*2.0f) * (v+t*0.5f) );
						const float angle = (3.141592653589793f*2.0f)*(t*0.005f + v);
						r.x = rect.x + 14 + (int)((a * t) * cosf(angle));
						r.y = rect.y + 14 + (int)((a * t) * sinf(angle));
						SDL_BlitSurface( GFX.stars[i&3], NULL, viewport->screen, &r );
					}
				}
			}
			else
			{
				char buf[64];
				SDL_BlitSurface( GFX.tiles[3], NULL, viewport->screen, &rect );
				sprintf(buf, "%d", this->timeout_counter+1 );

				GFX.font->PrintText( viewport->screen, 
						rect.x + viewport->tilewidth/2, 
						rect.y , buf );
			}
		}
		break;
	default: break;
	}

	if( bmp ) SDL_BlitSurface( bmp, NULL, viewport->screen, &rect );

}	

ObjectItem :: ObjectItem()
	:	oldx(0), oldy(0),
		life(3), keys(0), coins(0), score(0),
		update_counter(0), update_counter_max(8), 
		anim(0), anim_delay(0), anim_delay_max(300.200f),
		itemtype(0), timeout_counter(0), timeout_delay(0)
{
}

ObjectItem :: ~ObjectItem()
{

}

