#include <list>
#include <cassert>

#include <SDL.h>

#include "object_manager.h"
#include "object.h"
#include "viewport.h"

void ObjectManager :: Clear()
{
	object_list_t::iterator iter = list.begin();

	for( ; iter != list.end(); )
	{
		Object *obj = *iter;
		assert( obj );
		delete obj;
		iter = list.erase( iter );
	}
    list.clear();
	main_object = NULL;
}

void ObjectManager :: LinkObject( Object *obj )
{
	obj->parent = this;
	list.push_back( obj );
}

void ObjectManager :: Update(const float dt)
{
	// Check collisions:
	for(object_list_t::iterator outeriter = list.begin(); outeriter != list.end(); ++outeriter )
	{
		Object *outerobj = *outeriter;
		assert( outerobj );

		object_list_t::iterator inneriter = outeriter;
		++inneriter;
		for(; inneriter != list.end(); ++inneriter )
		{
			Object *innerobj = *inneriter;
			assert( innerobj );	
			outerobj->CheckCollision( innerobj );
		}
	}

	// Update game logic, AI, etc:
	for(object_list_t::iterator iter = list.begin(); iter != list.end(); ++iter )
	{
		Object *obj = *iter;
		assert( obj );
		obj->Update(dt);
	}

	// remove objects marked for deletion:
	for(object_list_t::iterator iter = list.begin(); iter != list.end();  )
	{
		Object *obj = *iter;
		assert( obj );
		if( obj->delete_me )
		{
			if( obj == main_object )
			{
				main_object = NULL;
			}
			delete obj;
			iter = list.erase( iter );
		}
		else
		{
			++iter;
		}
	}

	if( main_object == NULL )
	{
		main_object =  FindObject( ObjectTypes::Player );
	}
}

void ObjectManager :: UpdateAnims(const float dt)
{
	// Update game logic, AI, etc:
	for(object_list_t::iterator iter = list.begin(); iter != list.end(); ++iter )
	{
		Object *obj = *iter;
		assert( obj );
		obj->UpdateAnims(dt);
	}
}


void ObjectManager :: Draw(Viewport *viewport, const float alpha)
{
	for(object_list_t::iterator iter = list.begin(); iter != list.end(); ++iter )
	{
		Object *obj = *iter;
		assert( obj );
		obj->Draw(viewport, alpha);
	}
}

Object * ObjectManager :: FindObject(int type)
{
	for(object_list_t::iterator iter = list.begin(); iter != list.end(); ++iter )
	{
		Object *obj = *iter;
		assert( obj );
		if( obj->GetType() == type )
		{
			return obj;
		}
	}
	return NULL;
}


void ObjectManager :: ExplosionAt(int x, int y,int radius)
{
	for(object_list_t::iterator iter = list.begin(); iter != list.end(); ++iter )
	{
		Object *obj = *iter;
		assert( obj );

		if( (obj->x >= x - radius) && 
			(obj->y >= y - radius) && 
			(obj->x <= x + radius) && 
			(obj->y <= y + radius) )
		{
			obj->Death();
		}
	}
}

int ObjectManager :: GetObjectTypeCount(int type)
{
	int result = 0;
	for(object_list_t::iterator iter = list.begin(); iter != list.end(); ++iter )
	{
		Object *obj = *iter;
		assert( obj );
		if( obj->GetType() == type )
		{
			++result;
		}
	}
	return result;
}

int ObjectManager :: GetObjectSubTypeCount(int type, int subtype)
{
	int result = 0;
	for(object_list_t::iterator iter = list.begin(); iter != list.end(); ++iter )
	{
		Object *obj = *iter;
		assert( obj );
		if( (obj->GetType() == type) && (obj->GetSubType() == subtype) )
		{
			++result;
		}
	}
	return result;
}

ObjectManager :: ObjectManager(Map *owner)
	:	main_object(NULL)
{
	list.clear();
	map = owner;
}

ObjectManager :: ~ObjectManager()
{
	Clear();
}



