#include <cassert>
#include <cmath>

#include <SDL.h>

#include "map.h"

#include "object_manager.h"
#include "object.h"
#include "object_player.h"
#include "object_monster.h"

#include "viewport.h"
#include "gfx.h"
#include "snd.h"

class MonsterStates
{
public:
	enum
	{
		None = 0,
		Standing,
		Walking,
		Climbing,
		Falling,
		Teleport,
		Death,
			
		Last
	};
};

void Monster :: CollisionWith( Object * other )
{

}

// death by explosion:
void Monster :: Death()
{
	if( this->state != MonsterStates::Death )
	{
		SND.sound[6]->play();
		this->state = MonsterStates::Death;
	}
}

void Monster :: UpdateAnims(const float dt)
{
	anim_delay += dt;
	if( anim_delay >= anim_delay_max )
	{
		anim_delay -= anim_delay_max;
		anim = (anim + 1) & 7;
	}

	++update_counter;
	if( update_counter >= update_counter_max )
	{
		oldx = x;
		oldy = y;

		update_counter -= update_counter_max;
	}

}

void Monster :: Update(const float dt)
{
	anim_delay += dt;
	if( anim_delay >= anim_delay_max )
	{
		anim_delay -= anim_delay_max;
		anim = (anim + 1) & 7;
	}

	++update_counter;
	if( update_counter >= update_counter_max )
	{
		oldx = x;
		oldy = y;

		update_counter -= update_counter_max;
		assert( parent );

		Map *map = parent->GetMap();
		assert(map);

		if( state == MonsterStates::Falling )
		{
			++falling_count;
		}
		else
		{
			falling_count = 0;
		}

		if( last_teleport_counter>0 )
		{
			--last_teleport_counter;
		}
		else
		{
			last_teleport_x = -1;
			last_teleport_y = -1;
		}

		if( state == MonsterStates::Death )
		{
			this->delete_me = true;
		}
		else
		{
			// move forward direction of change direction:

			state = MonsterStates::None;

			bool has_move = false;
			int newx = x;
			int newy = y;
			int tiletype = map->GetTileTypeAt( x, y );

			switch( tiletype )	
			{
			case TileTypes::Ladder:	// Climbing:
				{
					int tiletype_above = map->GetTileTypeAt( x, y-1 );
					switch( tiletype_above )
					{
					case TileTypes::None:
					case TileTypes::Ladder:
					case TileTypes::Open:
					case TileTypes::OneWay_Left:
					case TileTypes::OneWay_Right:
					case TileTypes::Skull:
					case TileTypes::Teleport:
						has_move = true;
						newy = y - 1;
						state = MonsterStates::Climbing;
						break;
					default: break;
					};
				}
				break;
			case TileTypes::OneWay_Left:
				direction = -1;
				break;
			case TileTypes::OneWay_Right:
				direction = 1;
				break;
				/*
			case TileTypes::Skull:
				has_move = true;
				state = MonsterStates::Death;
				break;
				*/

			case TileTypes::Teleport:
				if( (last_teleport_x != x) && (last_teleport_y != y) )
				{
					newx = x; newy = y;
					if( map->GetTeleportDestination( &newx, &newy ) )
					{
						has_move = true;
						state = MonsterStates::Teleport;
						last_teleport_x = newx;
						last_teleport_y = newy;
						last_teleport_counter = 1;

						SND.sound[0]->play();
					}
				}
				break;
			default: break;
			}

			if( !has_move )	// Falling down:
			{
				int tiletype_below = map->GetTileTypeAt( x, y+1 );
				switch( tiletype_below )
				{
				case TileTypes::Open:
				case TileTypes::None:
				case TileTypes::OneWay_Left:
				case TileTypes::OneWay_Right:
				case TileTypes::Skull:
				case TileTypes::Teleport:				
					has_move = true;
					newy = y + 1;
					state = MonsterStates::Falling;
					if( falling_count == 3 )
					{
						SND.sound[2]->play();
					}
					break;
				default: break;
				};			

				if( !has_move )
				{
					int tiletype_left = map->GetTileTypeAt( x + direction, y );
					switch( tiletype_left )
					{
					case TileTypes::None:
					case TileTypes::Ladder:
					case TileTypes::Open:
					case TileTypes::Skull:
					case TileTypes::Teleport:
						has_move = true;
						break;
					case TileTypes::OneWay_Left:
						if( direction < 0 )
						{
							has_move = true;
						}
						break;
					case TileTypes::OneWay_Right:
						if( direction > 0 )
						{
							has_move = true;
						}
						break;
					default: break;
					};	

					if( !has_move )	// change direction:
					{
						direction = -direction;
						int tiletype_right = map->GetTileTypeAt( x + direction, y );
						switch( tiletype_right )
						{
						case TileTypes::None:
						case TileTypes::Ladder:
						case TileTypes::Open:
						case TileTypes::Skull:
						case TileTypes::Teleport:
							has_move = true;
							break;
						case TileTypes::OneWay_Left:
							if( direction < 0 )
							{
								has_move = true;
							}
							break;
						case TileTypes::OneWay_Right:
							if( direction > 0 )
							{
								has_move = true;
							}
							break;
						default: break;
						};	
					}

					if( has_move )
					{
						newx = x + direction;
						state = MonsterStates::Walking;
					}
				}
			}

			/*
			if( (falling_count>3) && (state != MonsterStates::Falling) )
			{
				newx = x;
				newy = y;
				state = MonsterStates::Death;
			}
			*/

			x = newx;
			y = newy;

			if( state == MonsterStates::Death )
			{
				SND.sound[6]->play();
			}
		}
	}
}

void Monster :: Draw(Viewport *viewport, const float alpha)
{
	const int direction_lookup[] = {0,0,1};
	SDL_Rect rect;
	SDL_Surface *bmp = NULL;

	float t = ((float)update_counter + (float)alpha) / (float)update_counter_max;
	if( t>1.0f) t = 1.0f;

	float xp = t * (float)(x - oldx) + oldx;
	float yp = t * (float)(y - oldy) + oldy;

	rect.x = viewport->offsetx + (int)((float)viewport->tilewidth  * xp);
	rect.y = viewport->offsety + (int)((float)viewport->tileheight * yp);
		
	switch( state )
	{
	case MonsterStates::Walking:
		{
			bmp = GFX.monster[ (anim&1) + 2*direction_lookup[1+direction] ];
		}
		break;
	case MonsterStates::None:
	case MonsterStates::Standing:
		bmp = GFX.monster[ 1 + 2*direction_lookup[1+direction] ];
		break;
	case MonsterStates::Climbing:
		bmp = GFX.monster[ (anim&1) + 2*direction_lookup[1+direction] ];
		break;
	case MonsterStates::Falling:
		bmp = GFX.monster[ (anim&1) + 2*direction_lookup[1+direction] ];
		break;

	case MonsterStates::Teleport:
		{
			SDL_Rect r;
			const int max_stars = 16;
			for(int i=0; i<max_stars; ++i)
			{
				const float angle = (3.141592653589793f*2.0f)*(t*0.125f + (float)i/(float)max_stars);
				r.x = rect.x + 14 + (int)(16.0f * cosf(angle));
				r.y = rect.y + 14 + (int)(16.0f * sinf(angle));
				SDL_BlitSurface( GFX.stars[0], NULL, viewport->screen, &r );
			}
		}
		break;
	case MonsterStates::Death:
			
		{
			SDL_Rect r;
			const int max_stars = 16;
			for(int i=0; i<max_stars; ++i)
			{
				const float angle = (3.141592653589793f*2.0f)*(t*0.25f + (float)i/(float)max_stars);
				r.x = rect.x + 14 + (int)(32.0f * t * cosf(angle));
				r.y = rect.y + 14 + (int)(32.0f * t * sinf(angle));
				SDL_BlitSurface( GFX.stars[i&1], NULL, viewport->screen, &r );
			}
		}

		break;
	}

	if(bmp) SDL_BlitSurface( bmp, NULL, viewport->screen, &rect );

}	

Monster :: Monster()
	:	oldx(0), oldy(0),
		life(3), keys(0), coins(0), score(0),
		update_counter(0), update_counter_max(10), 
		walk_speed(1), anim(0), anim_delay(0), anim_delay_max(300.200f),
		falling_count(0),last_teleport_counter(0),
		last_teleport_x(-1), last_teleport_y(-1)
{
	direction = -1;
}

Monster :: ~Monster()
{

}

