#include <math.h>

#include <SDL.h>
// #include <SDL_image.h>

#include <physfs.h>
#include "physfsrwops.h"

// #include <audiere.h>

#include "math_consts.h"

#include "game.h"
#include "maps.h"
#include "gamefont.h"

#include "mt19937ar.h"

namespace MadahGame
{

// Load image using physfs and converts into correct displayformat
SDL_Surface * LoadImage(const char *filename, const bool colorkey, const Uint32 key)
{
	SDL_Surface *s = NULL;
	SDL_RWops *rw = PHYSFSRWOPS_openRead( filename );
	if( rw )
	{
		s = SDL_LoadBMP_RW(rw,0);
			//IMG_Load_RW(rw,0);
		if(s)
		{
			if(colorkey)	SDL_SetColorKey(s, SDL_SRCCOLORKEY|SDL_RLEACCEL, key);
			SDL_Surface *temp = SDL_DisplayFormat(s);
			if(temp)
			{
				SDL_FreeSurface(s);
				s = temp;
			}
		}
		SDL_FreeRW(rw);
	}
	return s;
}

// Redraw entire screen:
void Game::Draw(SDL_Surface *screen, const float alpha)
{
	switch( screen_state )
	{
	case STATE_SCREEN_GAME:		Draw_Game(screen,alpha);	break;
	case STATE_SCREEN_MENU:		Draw_Menu(screen,alpha);	break;
	case STATE_SCREEN_GAMEEND:	Draw_GameEnd(screen,alpha);	break;
	case STATE_SCREEN_HELP:		Draw_Help(screen,alpha);	break;
	}
}


void Game::Draw_GameEnd(SDL_Surface *screen, const float alpha)
{
	const int yoff = 80;

	SDL_Rect rect;

	for(int p=0; p<=MAX_PLAYERS; p++)
	{
		rect.w = (Uint16)576;
		rect.h = (Uint16)2;
		rect.x = (Sint16)40;
		rect.y = (Sint16)(yoff + p * 64);

		SDL_FillRect(screen, &rect, graycolor );
	}

	// calc score
	int bestscore=-1;
	int winners=0;
	for(int p=0; p<MAX_PLAYERS; p++)
	{
		player[p]->score = player[p]->treasurecount * 8;
		if( player[p]->has_boots )		player[p]->score += 2;
		if( player[p]->has_binoculars )	player[p]->score += 1;

		if( player[p]->score > bestscore )
		{
			bestscore = player[p]->score;
		}
	}

	for(int p=0; p<MAX_PLAYERS; p++)
	{
		if( player[p]->score >= bestscore )
		{
			winners++;
		}
	}

	for(int p=0; p<MAX_PLAYERS; p++)
	{
		if( player[p]->active )
		{
			rect.x = (Sint16)32;
			rect.y = (Sint16)(yoff + p*64);

			SDL_BlitSurface( player[p]->gfx, NULL, screen, &rect );

			// Treasures:
			{
				char temp[64];
				sprintf(temp, "%d", player[p]->treasurecount);
				font->PrintText(screen, 128, yoff+p*64+16+8, temp );
				if( player[p]->treasurecount>0 )
				{
					int size = 72/player[p]->treasurecount;

					for(int j=0; j<player[p]->treasurecount; j++)
					{
						rect.x = (Sint16)(144 + j*size);
						rect.y = (Sint16)(yoff + p * 64);
						SDL_BlitSurface( tiletreasure, NULL, screen, &rect );
					}
				}
			}

			// More Items
			if( player[p]->has_boots )
			{
				rect.x = (Sint16)(128 + 64 + 16 + 64*1);
				rect.y = (Sint16)(yoff + p * 64);
				SDL_BlitSurface( tileboots, NULL, screen, &rect );
			}

			if( player[p]->has_binoculars )
			{
				rect.x = (Sint16)(128 + 64 + 16 + 64*2);
				rect.y = (Sint16)(yoff + p * 64);
				SDL_BlitSurface( tilebinoculars, NULL, screen, &rect );
			}

			if( (player[p]->score >= bestscore) && (bestscore>0) )
			{
				const char *winmsg[] = {"WINNER!","WINNER (DRAW)"};
				const char *s = winmsg[0];
				if( winners>1 ) s = winmsg[1];

				font->PrintText(screen, 128 + 64 + 16 + 64*4, yoff+p*64+16+8, s );
			}
		}
	}

	font->PrintText(screen, 128, 56, "Treasures:");
	font->PrintText(screen, 128 + 64 + 16 + 64*1 + 16, 56, "Items:");

	font->PrintText(screen, 8,8, "Game Summary:");

	font->PrintText(screen, 0,600-16, game_copyright);

}

void Game::Draw_Game(SDL_Surface *screen, const float alpha)
{
	SDL_Rect rect;

	// draw background tilemap:

	boardtile_t *ptr = map->tiles;
	for(int y=0; y<map->height; y++)
	{
		for(int x=0; x<map->width; x++)
		{
			int x1,y1;
			map->WorldToScreenCoord(x,y,&x1,&y1);

			rect.x = (Sint16)x1;
			rect.y = (Sint16)y1;

			if( ptr->explored )
			{
				switch( ptr->type )
				{
				case 0:
					SDL_BlitSurface( tileplain, NULL, screen, &rect );
					break;
				case 1:
					SDL_BlitSurface( tilegrass, NULL, screen, &rect );
					break;
				case 2:
					SDL_BlitSurface( tilewater, NULL, screen, &rect );
					break;
				}

				if( ptr->has_road )
				{
					if( ptr->roadmask != 0 )
					{
						for(int r=0; r<6; r++)
						{
							if( (ptr->roadmask & (1<<r)) )
							{
								SDL_BlitSurface( tileroad[r], NULL, screen, &rect );
							}
						}
					}
					SDL_BlitSurface( tileroad[6], NULL, screen, &rect );
				}

				if( ptr->has_treasure )
				{
					SDL_BlitSurface( tilequestion, NULL, screen, &rect );
				}
			}
			else if( ptr->disabled == false )
			{
				SDL_BlitSurface( tilebase, NULL, screen, &rect );
			}

			ptr++;
		}
	}

	// moving tiles

	// Static:
	for(int i=0; i<6; i++)
	{
		rect.x	 = (Sint16)(800 - PILE_TILE_RIGHT);
		rect.y	 = (Sint16)(64 - i*8);

		SDL_BlitSurface( tilegrass, NULL, screen, &rect );
	}

	// moving:
	for(int i=0; i<MAX_MOVINGTILES; i++)
	{
		if( movingtiles[i].active && (movingtiles[i].movecounter>0) )
		{
			float t = (float)((float)movingtiles[i].movecounter + alpha)/(float)movingtiles[i].movecountermax;

		float t2 = t;

		if( t2<0.5f )
		{
			t2 *= 2.0f;
			t2 *= t2;
			//t2 *= t2;
			t2 *= 0.5f;
		}
		else
		{
			t2 -= 0.5f;

			t2 *= 2.0f;
			t2 = 1.0f - t2;
			t2 *= t2;
			//t2 *= t2;
			t2 *= 0.5f;

			t2 = 1.0f - t2;
		}

			rect.x	 = (Sint16)((int)(t2 * (movingtiles[i].stopx - movingtiles[i].startx) ) + movingtiles[i].startx);
			rect.y	 = (Sint16)((int)(t2 * (movingtiles[i].stopy - movingtiles[i].starty) ) + movingtiles[i].starty);

			SDL_BlitSurface( tilegrass, NULL, screen, &rect );
		}
	}

	// players


	font->PrintText(screen, 800-144+8, 256-16, "Treasures:");
	for(int i=0; i<MAX_PLAYERS; i++)
	{
		if( player[i]->active )
		{
			player[i]->Draw(screen, alpha);
			// stats at the right screen

			{
				rect.x = (Sint16)(800-144-16);
				rect.y = (Sint16)(256 + i * 48);
				SDL_BlitSurface( player[i]->gfx, NULL, screen, &rect );

				if( player[i]->treasurecount>0 )
				{
					int size = 72/player[i]->treasurecount;

					for(int j=0; j<player[i]->treasurecount; j++)
					{
						rect.x = (Sint16)(800-144+24 + j*size);
						rect.y = (Sint16)(256 + i * 48);
						SDL_BlitSurface( tiletreasure, NULL, screen, &rect );
					}
				}
			}
		}
	}

	// selected mouse tile
	{
		int x1,y1,x2,y2, xc, yc;

		x1 = mousestate[0].x;
		y1 = mousestate[0].y;

		map->ScreenToWorldCoord(x1,y1, &x2, &y2);
		map->WorldToScreenCoord(x2,y2,&x1,&y1);

		rect.x = (Sint16)x1;
		rect.y = (Sint16)y1;
		SDL_BlitSurface( tileselect, NULL, screen, &rect );

		Player *p = player[this->current_player];
		if( p && p->active )
		{
			map->WorldToScreenCoordCenter(p->x,p->y,&xc,&yc);

			// stats for this player
			{
				char temp[64];

				sprintf(temp, "%d / %d", p->movepointsleft, p->explorepointsleft );
				font->PrintText(screen, xc-40,yc-80, temp);
			}

			if( p->ready )
			{
				xc -= tiledot->w/2;
				yc -= tiledot->h/2;

				{
					const int numdots = 12;
					for(int i=0; i<numdots; i++)
					{
						float t = ( ((float)i + ((float)(framecounter&0x07)+alpha)/8.0f) / (float)numdots ) * 2.0f * (float)MATH_PI;

						rect.x	= (Sint16)(xc + (int)((float)map->hexradius * cosf(t)));
						rect.y	= (Sint16)(yc + (int)((float)map->hexradius * sinf(t)));

						SDL_BlitSurface( tiledot, NULL, screen, &rect );
					}
				}

				if( p == active_player )
				{
					// dots from player to mouse
					bool validpoint = false;
					int distance = map->GetDistance(p->x,p->y,x2,y2);

					validpoint = (!IsBlockedTile(x2,y2)) && (distance==1);

					if( distance == 0 )
					{
						font->PrintText(screen, mousestate[0].x, mousestate[0].y+16, "Pass");
					}
					else if( validpoint )
					{
						boardtile_t *tile = &map->tiles[y2*map->width+x2];
						int xc2, yc2;
						map->WorldToScreenCoordCenter(x2,y2,&xc2,&yc2);
						xc2 -= tiledot->w/2;
						yc2 -= tiledot->h/2;

						if( tile->explored )
						{
							if( p->movepointsleft>0 )
							{
								const int numdots = 6;
								for(int i=0; i<numdots; i++)
								{
									float t		= ( ((float)i + ((float)(framecounter&0x07)+alpha)/8.0f) / (float)numdots );
									float t2	= ( ((float)i + ((float)(framecounter&0x07)+alpha)/8.0f) / (float)numdots )*2.0f*(float)MATH_PI;


									rect.x	= (Sint16)(xc	+ (int)(t*(float)(xc2 - xc) ));//+ (float)map->hexradius*0.125f*sinf(t2));
									rect.y	= (Sint16)(yc	+ (int)(t*(float)(yc2 - yc) + (float)map->hexradius*0.125f*cosf(t2)));

									SDL_BlitSurface( tiledot, NULL, screen, &rect );
								}
								font->PrintText(screen, mousestate[0].x, mousestate[0].y+16, "Move");
							}
						}
						else	// unexplored
						{
							if( p->explorepointsleft>0 )
							{
								const int numdots = 6;
								for(int i=0; i<numdots; i++)
								{
									float t = ( ((float)i + ((float)(framecounter&0x07)+alpha)/8.0f) / (float)numdots ) * 2.0f * (float)MATH_PI;
									float t2 = 0.25f+ 0.125f*cosf(t*5.0f);

									rect.x	= (Sint16)(xc2	+ (int)((float)map->hexradius * cosf(t) * t2));
									rect.y	= (Sint16)(yc2	+ (int)((float)map->hexradius * sinf(t) * t2));

									SDL_BlitSurface( tiledot, NULL, screen, &rect );
								}
								font->PrintText(screen, mousestate[0].x-32, mousestate[0].y+16, "Explore");
							}
						}

					}

				}
			}
		}
		/*
		int tx[6],ty[6];
		int count = map->GetNearbyTiles(x2,y2,&tx[0],&ty[0]);

		map->WorldToScreenCoordCenter(x2,y2,&xc,&yc);
		for(int i=0; i<count; i++)
		{
			map->WorldToScreenCoordCenter(tx[i],ty[i],&x1,&y1);

			for(int j=0; j<5; j++)
			{
				float t  = (float)((j+3))/6.0f;
				float t2 = (float)((j)-(framecounter&0x07)/8.0f)/6.0f;

				float u  = 4.0f*cosf((t2)*2.0f*3.141592653589793238462643383279502884197169399375105820974944592f);

				rect.x = (int)(t2*(float)(x1-xc) + xc);
				rect.y = (int)(t2*(float)(y1-yc) + yc + u);

				SDL_BlitSurface( tiledot, NULL, screen, &rect );
			}
		}
		*/
	}

	if( foundtreasure.active )
	{
		float t = ((float)foundtreasure.count+alpha) / (float)foundtreasure.countmax;
		//float t2 = (float)((t)-(framecounter&0x07)/8.0f);

		float u  = 6.0f*cosf((t)*4.0f*(float)MATH_PI);

		int trx1 = foundtreasure.wsx + (int)(u);
		int try1 = foundtreasure.wsy + (int)(-128.0f*t);

		int size = 0;

		if( foundtreasure.player->treasurecount>0 )
		{
			size = 72/(1+foundtreasure.player->treasurecount);
		}

		int trx2 =  800-144+24 + (foundtreasure.player->treasurecount) * size;
		int try2 = 256 + foundtreasure.player->index*48;

		if( foundtreasure.treasure_type == 0)
		{
			float t2 = t;

			if( t2<0.5f )
			{
				t2 *= 2.0f;
				t2 *= t2;
				//t2 *= t2;
				t2 *= 0.5f;
			}
			else
			{
				t2 -= 0.5f;

				t2 *= 2.0f;
				t2 = 1.0f - t2;
				t2 *= t2;
				//t2 *= t2;
				t2 *= 0.5f;

				t2 = 1.0f - t2;
			}
			if(t2<0.0f) t2=0.0f;

			rect.x	= (Sint16)(trx1 + (int)(t2 * (float)(trx2 - trx1)));
			rect.y	= (Sint16)(try1 + (int)(t2 * (float)(try2 - try1)));
		}
		else
		{
			rect.x = (Sint16)trx1;
			rect.y = (Sint16)(foundtreasure.wsy + (int)(-64.0f*t));
		}
		SDL_Surface *st[] = {tiletreasure, tileboots, tilebinoculars, tilespider, tilebear, tilecrow};

		SDL_BlitSurface( st[foundtreasure.treasure_type], NULL, screen, &rect );

		font->PrintText(screen, foundtreasure.textx, foundtreasure.texty - (int)(24.0f*t), foundtreasure.msg );
	}

	font->PrintText(screen, 0,600-16, game_copyright);
	/*
	rect.x = (int)( (x - oldx)*alpha + oldx ) ;
	rect.y = (int)( (y - oldy)*alpha + oldy );

	SDL_BlitSurface( bmp1, NULL, screen, &rect );
	*/
}

// Update game logic:
// dt kan man egentligen strunta i
void Game::Update(const float dt)
{
	switch( screen_state )
	{
	case STATE_SCREEN_GAME:		Update_Game(dt);	break;
	case STATE_SCREEN_MENU:		Update_Menu(dt);	break;
	case STATE_SCREEN_GAMEEND:	Update_GameEnd(dt);	break;
	case STATE_SCREEN_HELP:		Update_Help(dt);	break;
	}
}


void Game::Update_GameEnd(const float dt)
{
	if( this->keypressed[SDLK_ESCAPE] || this->keypressed[SDLK_SPACE] )
	{
		this->screen_state = STATE_SCREEN_MENU;
	}

	/*
	if( this->keypressed[SDLK_q] )
	{
		this->screen_state = STATE_SCREEN_GAME;
	}
	*/
	/*
	if( this->keypressed[SDLK_w] )
	{
		player[0]->has_boots =true;
		player[0]->has_binoculars = true;
		player[0]->treasurecount = 10;

		player[1]->has_boots =true;
		player[2]->has_binoculars = true;

		player[1]->treasurecount = 7;
		player[2]->treasurecount = 5;
	}
	*/

	framecounter++;
	framecounter &= 0x3FFFFFFF;

	// Clear keyboard/mouse
	ResetStates();
}

void Game::Update_Game(const float dt)
{
	// check keyboard/mouse and move player

	if( mousestate[1].mbpressed )
	{
		Player *newselection = NULL;
		int tx,ty;

		map->ScreenToWorldCoord(mousestate[1].x, mousestate[1].y, &tx, &ty );
		if( (tx>=0) && (ty>=0) && (tx<map->width) && (ty<map->height) )
		{
			// kolla om man tryckt p en annan player:
			for(int i=0; i<MAX_PLAYERS; i++)
			{
				if( player[i]->active )
				{
					if( (player[i]->x == tx) && (player[i]->y == ty) )
					{
						newselection = player[i];
						break;
					}
				}
			}

			if(newselection)
			{
				// man har tryckt p en annan player
				if( active_player )
				{
					// man har tryckt p sig sjlv:
					if( active_player == newselection )
					{
						active_player->pass = true;
					}
					else
					{
						active_player->Hotspot(tx,ty, newselection);
					}
				}
				else
				{
					// detta blir den nya selected:
					selected_player = newselection;
				}
			}
			else
			{
				// man har tryckt p en position p kartan
				if( active_player )
				{
					if( map->GetDistance(active_player->x, active_player->y, tx, ty ) == 1 )
					{
						boardtile_t *tile = &map->tiles[ty*map->width+tx];

						if( tile->explored )
						{
							if( !IsBlockedTile(tx, ty) )
							{
								active_player->HotspotMove(tx,ty);
								//active_player->movepointsleft--;
							}
						}
						else
						{
							active_player->HotspotExplore(tx,ty);
							//AddNewTileAt( tx, ty, 0 );
							//active_player->explorepointsleft--;
						}
					}
				}
			}
		}
	}
	else if( mousestate[3].mbpressed )
	{
		// pass this turn
		if( active_player )
		{
			active_player->pass = true;

		}
	}

	// AI etc

	// players

	if( game_isdone )
	{
		//game_isdone = false;
		if( current_player >= MAX_PLAYERS )	current_player = 0;

		int i=0;
		while( (player[current_player]->active == false) )
		{
			current_player++;
			if( current_player >= MAX_PLAYERS )	current_player = 0;
			i++;
			if( i>=MAX_PLAYERS ) break;
		}

		if( player[current_player]->active )
		{
			if( player[current_player]->human)
			{
				this->active_player = player[current_player];
			}

			player[current_player]->Update(dt);
			if( player[current_player]->isdone )
			{
				player[current_player]->isdone = false;
				player[current_player]->pass   = false;

				game_isdone = false;
				CheckPlayerMove( player[current_player] );

				current_player++;
				if( current_player >= MAX_PLAYERS )	current_player = 0;
			}
		}
	}
	//else
	{
		// kr spel-logik
		// t ex flytta in nya tiles
		// och allt som sker efter att alla spelare flyttat en gng

		DoGameLogic();

		//game_isdone = true;
	}

	/*
	for(int i=0; i<MAX_PLAYERS; i++)
	{
		if( player[i]->active )
		{
			player[i]->Update(dt);
		}
	}
	*/

//	oldx = x;
//	oldy = y;

//	x += 15;
//	y += 15;

//	if( x>=640 ) x -= 640;
//	if( y>=480 ) y -= 480;

	if( this->keypressed[SDLK_ESCAPE] || this->keypressed[SDLK_q] )
	{
		this->screen_state = STATE_SCREEN_GAMEEND;
	}

	if( this->keypressed[SDLK_h] || this->keypressed[SDLK_F1] )
	{
		this->help_prev_state = screen_state;
		screen_state = STATE_SCREEN_HELP;
	}

	this->CheckGameOver();

	// Clear keyboard/mouse
	ResetStates();

	framecounter++;
	framecounter &= 0x3FFFFFFF;
}

void Game::CheckPlayerMove( Player *p )
{
	// kolla om man rrt sig in p outforskat omrde
	int numtiles;
	//int count = map->GetNearbyTiles(p->x, p->y, &tx[0], &ty[0]);

	numtiles =0;
	{
		boardtile_t *tile = &map->tiles[p->y*map->width+p->x];
		if( tile->explored == false )
		{
			AddNewTileAt( p->x, p->y, numtiles );
			numtiles++;
		}
		else
		{
			// kolla om man gtt p ngot speciellt etc
			if( tile->has_treasure )
			{
				AddPlayerFoundTreasure( p, p->x, p->y, tile );

				p->state = 3;
				p->movecounter = 0;
				p->movecountermax = foundtreasure.countmax;
			}
		}
	}
}

void Game::AddPlayerFoundTreasure(Player *p, int px, int py, boardtile_t *tile)
{
	foundtreasure.count			=	0;
	foundtreasure.countmax		=	16;

	foundtreasure.tilex	= px;
	foundtreasure.tiley	= py;

	foundtreasure.tileptr	= tile;
	tile->has_treasure	= false;

	int x1,y1;

	map->WorldToScreenCoord(px,py,&x1,&y1);

	foundtreasure.player = p;
	foundtreasure.wsx	= x1;
	foundtreasure.wsy	= y1;

	foundtreasure.textx	= x1 - 16;
	foundtreasure.texty	= y1 - 64;

	foundtreasure.treasure_type = genrand_int31() % 10;
	if( foundtreasure.treasure_type > 5 ) foundtreasure.treasure_type = 0;

	const char *msg[] = {	"A Treasure!",
							"Nice Boots!\ngives +1 walk",
							"Binoculars!\ngives +1 explore",
							"Bitten by a Spider!\nMust wait one turn",
							"A hungry Bear eats\nBoots & Binoculars!",
							"A black Crow\nsteals Treasure!\n"
						};

	if( foundtreasure.treasure_type > 0 )
	{
		foundtreasure.countmax += 16;
	}

	sprintf(foundtreasure.msg, msg[foundtreasure.treasure_type] );

	//p->treasurecount++;

	foundtreasure.active	= true;
}

bool Game::IsBlockedTile(const int x, const int y)
{
	// Check terrain:
	if( map->IsBlockedTile(x,y) )
	{
		return true;
	}
	// Check players:
	for(int i=0; i<MAX_PLAYERS; i++)
	{
		if( player[i]->active && (player[i]->x==x) && (player[i]->y==y) )
		{
			return true;
		}
	}
	return false;
}

void Game::AddNewTileAt( int x, int y, int index )
{
	int freeindex;
	for(freeindex=0; freeindex<MAX_MOVINGTILES; freeindex++)
	{
		if(movingtiles[freeindex].active == false )
		{
			movingtiles[freeindex].active = true;
			movingtiles[freeindex].worldx	= x;
			movingtiles[freeindex].worldy	= y;

			map->WorldToScreenCoord(x,y, &movingtiles[freeindex].stopx, &movingtiles[freeindex].stopy );

			movingtiles[freeindex].startx = 800 - PILE_TILE_RIGHT;
			movingtiles[freeindex].starty = 64 - 6*8;

			movingtiles[freeindex].movecounter		=	-index*4;
			movingtiles[freeindex].movecountermax	=	8;
			break;
		}
	}
}

void Game::DoGameLogic()
{
	bool stillmoving = false;

	// Treasure messages first

	if( foundtreasure.active )
	{
		stillmoving	= true;

		foundtreasure.count++;
		if( foundtreasure.count >= foundtreasure.countmax )
		{
			if( foundtreasure.tileptr )
			{
				foundtreasure.tileptr->has_treasure	= false;
			}
			foundtreasure.active = false;

			switch( foundtreasure.treasure_type )
			{
			case 0: foundtreasure.player->treasurecount++;			break;
			case 1: foundtreasure.player->has_boots			= true;
					foundtreasure.player->movepointsleft++;
					break;
			case 2: foundtreasure.player->has_binoculars	= true;
					foundtreasure.player->explorepointsleft++;
					break;
			case 3: foundtreasure.player->has_spider		= true;
					foundtreasure.player->pass = true;
					break;
			case 4:	foundtreasure.player->has_boots			= false;
					foundtreasure.player->has_binoculars	= false;
					foundtreasure.player->pass = true;
					break;
			case 5:	foundtreasure.player->treasurecount--;
					if( foundtreasure.player->treasurecount <0 )
					{
						foundtreasure.player->treasurecount = 0;
					}
					foundtreasure.player->pass = true;
					break;
			}
		}
	}
	else
	{
		for(int i=0; i<MAX_MOVINGTILES; i++)
		{
			if( movingtiles[i].active )
			{
				stillmoving = true;

				movingtiles[i].movecounter++;
				if( movingtiles[i].movecounter >= movingtiles[i].movecountermax )
				{
					movingtiles[i].active = false;

					boardtile_t * tile = &map->tiles[movingtiles[i].worldy*map->width + movingtiles[i].worldx];
					tile->explored = true;

					tile->type = 1;

					if( (genrand_int32() & 0x7) == 0 )	tile->type = 2;
					//if( (genrand_int32() & 0xF) == 0 )

					if( (tile->type == 1) )
					{
						if( (genrand_int32() & 0x1) == 0 )
						{
							tile->has_treasure		= true;
							tile->treasure_hidden	= true;
						}

						if( (genrand_int32() & 0x1) == 0 )
						{
							tile->type = 0;
							if( ((genrand_int32() & 0x1) == 0)  )
							{
								//tile->type = 0;
								tile->has_road = true;
								map->UpdateRoadMask();
							}
						}
					}

				}
			}
		}
	}

	if(stillmoving == false )
	{
		game_isdone = true;
		map->UpdateRoadMask();
	}
}


// Input handling:  SDLinput -> game wrapper

void Game::InitStates()
{
	for(int i=0; i<MAX_KEYSTATES; i++)
	{
		keypressed[i]	= false;
		keydown[i]		= false;
	}

	for(int i=0; i<MAX_MOUSESTATES; i++)
	{
		mousestate[i].mbpressed	= false;
		mousestate[i].button	= 0;
		mousestate[i].mbdown	= false;
		mousestate[i].moved		= false;
		mousestate[i].x			= 0;
		mousestate[i].y			= 0;
	}
}

void Game::ResetStates()
{
	for(int i=0; i<MAX_KEYSTATES; i++)
	{
		keypressed[i] = false;
	}

	for(int i=0; i<MAX_MOUSESTATES; i++)
	{
		mousestate[i].mbpressed	= false;
	}
}

void Game::KeyDown		(SDL_keysym *key)
{
	int i = key->sym;
	if( i < MAX_KEYSTATES )
	{
		keypressed[i]	= true;
		keydown[i]		= true;
	}
}

void Game::KeyUp		(SDL_keysym *key)
{
	int i = key->sym;
	if( i < MAX_KEYSTATES )
	{
		keydown[i]		= false;
	}
}

void Game::MouseMove	(SDL_MouseMotionEvent *motion)
{
	mousestate[0].moved = true;
	mousestate[0].x	= motion->x;
	mousestate[0].y = motion->y;
}

void Game::MouseDown	(SDL_MouseButtonEvent *button)
{
	int i = button->button;
	if( (i>=1) && (i<MAX_MOUSESTATES) )
	{
		mousestate[i].mbpressed = true;
		mousestate[i].mbdown	= true;
		mousestate[i].x			= button->x;
		mousestate[i].y			= button->y;
	}
}

void Game::MouseUp		(SDL_MouseButtonEvent *button)
{
	int i = button->button;
	if( (i>=1) && (i<MAX_MOUSESTATES) )
	{
		mousestate[i].mbdown	= false;
		mousestate[i].x			= button->x;
		mousestate[i].y			= button->y;
	}
}

// constructor/destructor
Game::Game()
	: framecounter(0), should_quit(false), movingtilescount(0), game_isdone(false)
{
	// Virtual filesystem:
	PHYSFS_init(NULL);
	PHYSFS_addToSearchPath(".", 0);
	PHYSFS_addToSearchPath("gamedata.dat", 1);

	// Sound:
	//device = audiere::OpenDevice();

	map = new BoardMap();
	map->AllocMap(11,8);
	map->SetHexRadiusHeight(37,32);

	for(int i=0; i<map->width; i+=2)
	{
		map->tiles[i].disabled	= true;
	}

	map->SetScreenSize(800,600);


	for(int i=0; i<MAX_PLAYERS; i++)
	{
		player[i] = new Player();
		player[i]->game = this;
		player[i]->map	= map;
	}

	tilebase	= LoadImage("gamedata/boardtile.bmp",	true, 0xFF00FF);
	tileselect	= LoadImage("gamedata/selected.bmp",	true, 0xFF00FF);
	tilegrass	= LoadImage("gamedata/grass1.bmp",		true, 0xFF00FF);
	tileplain	= LoadImage("gamedata/grass2.bmp",		true, 0xFF00FF);
	tilewater	= LoadImage("gamedata/water.bmp",		true, 0xFF00FF);
	tileplayer	= LoadImage("gamedata/player.bmp",		true, 0xFF00FF);
	tiledot 	= LoadImage("gamedata/dot6b.bmp",		true, 0xFF00FF);
	tiletreasure= LoadImage("gamedata/treasure.bmp",	true, 0xFF00FF);

	tileboots		= LoadImage("gamedata/boots.bmp",		true, 0xFF00FF);
	tilebinoculars	= LoadImage("gamedata/binoculars.bmp",	true, 0xFF00FF);
	tilespider		= LoadImage("gamedata/spider.bmp",		true, 0xFF00FF);
	tilebear		= LoadImage("gamedata/bear.bmp",		true, 0xFF00FF);
	tilecrow		= LoadImage("gamedata/crow.bmp",		true, 0xFF00FF);

	// so that gray lines are actually gray on all bitdepths (8/16/24+)
	graycolor = 0x404040;
	{
		SDL_Surface *s = SDL_GetVideoSurface();
		if(s)
		{
			graycolor	=	SDL_MapRGB(s->format, 0x40,0x40,0x40);
		}
	}

	tilequestion= LoadImage("gamedata/question.bmp",	true, 0xFF00FF);
	for(int i=0; i<7; i++)
	{
		char temp[64];
		sprintf(temp, "gamedata/road%d.bmp", i );
		tileroad[i]	= LoadImage(temp,	true, 0xFF00FF);
	}

	for(int i=0; i<MAX_PLAYERS; i++)
	{
		char temp[64];
		sprintf(temp, "gamedata/player%d.bmp", i );
		player[i]->gfx	= LoadImage(temp,	true, 0xFF00FF);

	}

	font = new GameFont();
	font->SetSDLSurface( LoadImage("gamedata/font.bmp",		true, 0x000000) );
	font->SetCharWidthHeight( 16, 16 );
	font->SetKerning(12);

/*
	sound = audiere::OpenSound(device, "music.ogg", true);
	sound->setRepeat(true);
	sound->setVolume(0.5f);
	sound->play();
*/
	InitStates();
	//x=oldx=y=oldy=0;


	for(int i=0; i<4; i++)
	{
		player[i]->index	= i;
		player[i]->active	= true;
		player[i]->human	= false;
		player[i]->x = 1+(i%2)*8;
		player[i]->y = 1+(i/2)*5;

		boardtile_t *t = &map->tiles[player[i]->y*map->width + player[i]->x];
		t->explored = true;
		t->type = 0;
		t->has_road = true;

		player[i]->isdone = false;
	}

	player[0]->human	= true;

	//memset( &foundtreasure, 0, sizeof(foundtreasure_t) );

	foundtreasure.active	=	false;
	foundtreasure.count		=	0;
	foundtreasure.countmax	=	0;

	active_player = player[0];
	current_player = 0;

	for(int freeindex=0; freeindex<MAX_MOVINGTILES; freeindex++)
	{
		movingtiles[freeindex].active = false;
	}

	InitNewGame();

	screen_state = STATE_SCREEN_MENU;
}

Game::~Game()
{
	PHYSFS_deinit();
//	sound	= 0;
//	device	= 0;

	delete map;
	delete font;

	for(int i=0; i<MAX_PLAYERS; i++)
	{
		delete player[i];
	}
}

// Game logic:

void Game::InitNewGame()
{
	map->ClearMap();

	for(int i=0; i<map->width; i+=2)
	{
		map->tiles[i].disabled	= true;
	}

	for(int i=0; i<MAX_PLAYERS; i++)
	{
		player[i]->Reset();
		player[i]->index	= i;
		player[i]->x = 1+(i%2)*8;
		player[i]->y = 1+(i/2)*5;
		player[i]->isdone = false;
		player[i]->map  = map;
		player[i]->game = this;
	}

	player[4]->x = (player[0]->x + player[1]->x)/2;
	player[4]->y = player[0]->y;

	player[5]->x = (player[2]->x + player[3]->x)/2;
	player[5]->y = player[2]->y;

	for(int i=0; i<MAX_PLAYERS; i++)
	{
		if( player[i]->active )
		{
			boardtile_t *t = &map->tiles[player[i]->y*map->width + player[i]->x];
			t->explored = true;
			t->type		= 0;
			t->has_road = true;
		}
	}

	//memset( &foundtreasure, 0, sizeof(foundtreasure_t) );

	foundtreasure.active	= false;
	foundtreasure.count		= 0;
	foundtreasure.countmax	= 0;

	active_player = player[0];
	current_player = 0;

	for(int freeindex=0; freeindex<MAX_MOVINGTILES; freeindex++)
	{
		movingtiles[freeindex].active = false;
	}

	movingtilescount = 0;

	this->game_isdone = false;
	this->should_quit = false;

	this->framecounter = 0;

	ResetStates();
}

void Game::CheckGameOver()
{
	boardtile_t *ptr = map->tiles;
	int i, len = map->width*map->height;

	for(i=0; i<len; i++)
	{
		if( (ptr->disabled==false) && ((ptr->explored==false) || (ptr->has_treasure==true) ))
		{
			return;
		}
		ptr++;
	}

	// all tiles explored + all treasures taken

	this->screen_state = STATE_SCREEN_GAMEEND;
}

}
